from docplex.cp.model import *
import xml.etree.ElementTree as ET

from typing import List

from auxiliary import AuxiliaryStruct
from Parameter import Parameter
from customTypes import *
from log import logger
from patterns import *
from Slot import SlotReq
from utils import rOr


class Locali:
    def __init__(self, nome:str, dayFrom:Day, fasciaOrariaFrom:FasciaOraria, dayTo:Day, fasciaOrariaTo:FasciaOraria, quantita:int, 
                 tipoLocale:TipoLocale, capienzaLocale:CapienzaLocale = CapienzaLocale.NonDisponibile,
                 preseElettriche:PreseElettriche = PreseElettriche.Sconosciuto):
        '''Per gestire la disponibilità di un TipoLocale in un certo arco temporale'''
        self.nome:str = nome
        self.dayFrom:Day = dayFrom
        self.fasciaOrariaFrom:FasciaOraria = fasciaOrariaFrom
        self.dayTo:Day = dayTo
        self.fasciaOrariaTo:FasciaOraria = fasciaOrariaTo
        self.tipoLocale:TipoLocale = tipoLocale
        self.capienzaLocale:CapienzaLocale = capienzaLocale
        self.quantita:int = quantita
        self.preseElettriche:PreseElettriche = preseElettriche
        self.log:logger = logger()
        
    def __str__(self):
        res = self.nome + " " + str(self.tipoLocale) + " "
        if self.capienzaLocale != CapienzaLocale.NonDisponibile:
            res += str(self.capienzaLocale)
        res += " quantità: " + str(self.quantita) + " Da: " + str(self.dayFrom) + "-" + str(self.fasciaOrariaFrom) + " A: " + str(self.dayTo) + "-" + str(self.fasciaOrariaTo)
        return res
        
    def getQuantitaDisponibile(self, day:Day, fasciaOraria:FasciaOraria, tipoLocale:TipoLocale, 
                               capienzaLocale:CapienzaLocale = CapienzaLocale.NonDisponibile, preseElettriche:PreseElettriche = PreseElettriche.Sconosciuto):
        '''Return:\n
            0 se l'istanza corrente non rappresenta Locali di quel TipoLocale o se non si è all'interno della sua fascia temporale
            else: quantita (int)'''
        if dataCmp(fasciaOraria, day, self.fasciaOrariaFrom, self.dayFrom) >= 0 and dataCmp(fasciaOraria, day, self.fasciaOrariaTo, self.dayTo) <= 0:
            # corretto arco temporale
            if tipoLocale == self.tipoLocale and capienzaLocale == self.capienzaLocale:
                if preseElettriche == PreseElettriche.Si and self.preseElettriche == preseElettriche:
                    return self.quantita
                elif preseElettriche != PreseElettriche.Si:
                    return self.quantita
        return 0
            


class LocaliHandler(metaclass=SingletonMeta):
    def __init__(self):
        '''Si occupa di caricare da xml i dati riguardanti i LocaliAllocati nei vari slot temporali e di generare le varie strutture da cui
        ricavare poi gli hard constraint per limitare in modo accettabile l'allocazione dei vari Slot tenendo conto della disponibilità di Aule'''
        self.AUX:AuxiliaryStruct = AuxiliaryStruct()
        self.PARAM:Parameter = Parameter()
        self.log:logger = logger()
        self.debug:bool = False
        self.stats:bool = True
        self.listLocali:List[Locali] = list()
        
        self.listSlot_ACSLAB:List[int] = list()
        self.listSlot_LABINF:List[int] = list()
        self.listSlot_Laboratorio:List[int] = list()
        self.listSlot_LADISPE:List[int] = list()
        self.listSlot_LAIB:List[int] = list()
        self.listSlot_LED:List[int] = list()
        self.listSlot_LED1:List[int] = list()
        self.listSlot_LED2:List[int] = list()
        self.listSlot_Aula5T:List[int] = list()

        # 0 levels extended
        self.listSlot_AulePiccole:List[int] = list()
        self.listSlot_AuleMedie:List[int] = list()
        self.listSlot_AuleMedioGrandi:List[int] = list()
        self.listSlot_AuleGrandi:List[int] = list()
        
        # 1 level extended
        self.listSlot_Aule_PiccoleMedie:List[int] = list()
        self.listSlot_Aule_MedieMedioGrandi:List[int] = list()
        self.listSlot_Aule_MedioGrandiGrandi:List[int] = list()
        
        # 2 levels extended
        self.listSlot_Aule_PiccoleMedieMedioGrandi:List[int] = list()
        self.listSlot_Aule_MedieMedioGrandiGrandi:List[int] = list()
        
        # Slot che necessitano di prese elettriche, con i relativi livelli di espansione
        # 0 levels extended
        self.listSlot_PreseElettriche_AulePiccole:List[int] = list()
        self.listSlot_PreseElettriche_AuleMedie:List[int] = list()
        self.listSlot_PreseElettriche_AuleMedioGrandi:List[int] = list()
        self.listSlot_PreseElettriche_AuleGrandi:List[int] = list()
        
        # 1 level extended
        self.listSlot_PreseElettriche_Aule_PiccoleMedie:List[int] = list()
        self.listSlot_PreseElettriche_Aule_MedieMedioGrandi:List[int] = list()
        self.listSlot_PreseElettriche_Aule_MedioGrandiGrandi:List[int] = list()
        
        # 2 levels extended
        self.listSlot_PreseElettriche_Aule_PiccoleMedieMedioGrandi:List[int] = list()
        self.listSlot_PreseElettriche_Aule_MedieMedioGrandiGrandi:List[int] = list()
        

    def loadXmlLocaliAllocati(self, fileXmlLocaliAllocati:str):
        '''Carica il file xml contenente i LocaliAllocati nei vari intervalli temporali'''
        tree = ET.parse(fileXmlLocaliAllocati)
        root = tree.getroot()   
        self.log.info_log("LocaliHandler.loadXmlLocaliAllocati(): load Locali")
        
        for GruppoLocali in root:
            nome:str = ""
            tipoLocale:TipoLocale = TipoLocale.Sconosciuto
            capienza:CapienzaLocale = CapienzaLocale.NonDisponibile
            preseElettriche:PreseElettriche = PreseElettriche.No
            quantita:int = 0
            dayF:str = ""
            dayT:str = ""
            hF:str = ""
            hT:str = ""
            for cc in GruppoLocali:
                if cc.tag == "Nome":
                    nome = cc.text
                elif cc.tag == "Locale":
                    tipoLocale = getTipoLocaleFromString(cc.text)
                elif cc.tag == "Capienza":
                    capienza = getCapienzaLocaleFromString(cc.text)
                elif cc.tag == "PreseElettriche":
                    preseElettriche = getPreseElettricheFromString(cc.text)
                elif cc.tag == "Quantita":
                    quantita = int(cc.text)
                elif cc.tag == "Da":
                    for ccs in cc:
                        if ccs.tag == "Giorno":
                            dayF = ccs.text
                        elif ccs.tag == "FasciaOraria":
                            hF = ccs.text
                elif cc.tag == "A":
                    for ccs in cc:
                        if ccs.tag == "Giorno":
                            dayT = ccs.text
                        elif ccs.tag == "FasciaOraria":
                            hT = ccs.text 
            d1,h1 = getDayFasciaOrariaFromString(dayF, hF)
            d2,h2 = getDayFasciaOrariaFromString(dayT, hT)
            
            locali:Locali = Locali(nome, d1, h1, d2, h2, quantita, tipoLocale, capienza, preseElettriche)
            if self.debug:
                self.log.info_log(locali)
            self.listLocali.append(locali)
            
        
    def getLimit(self, day:int, fasciaOraria:int, tipoLocale:TipoLocale, capienzaLocale:CapienzaLocale = CapienzaLocale.NonDisponibile,
                 preseElettriche:PreseElettriche = PreseElettriche.Sconosciuto) -> int:
        '''Ritorna il limite per la il TipoLocale nel (fasciaOraria,day) desiderato'''
        strDay, strHour = getStringsFromDayHour(day, fasciaOraria)
        d, f = getDayFasciaOrariaFromString(strDay, strHour)
        
        return sum(map(lambda q: q.getQuantitaDisponibile(d, f, tipoLocale, capienzaLocale, preseElettriche), self.listLocali))
        
    def createListsLocaliOfSlot(self):
        '''Basandosi su self.AUX.pianoAllocazione crea le varie liste raggruppanti tutti gli Slot che necessitano di un certo tipo di Locale'''
        for slotId in self.AUX.map_strSlotId_to_idSlot.values():
            slot = self.AUX.pianoAllocazione[slotId]
            
            # non tengo conto degli Insegnamenti del primo anno e di quelli esterni ai collegi esaminati
            if int(self.AUX.list_Insegnamenti[slot.idInsegnamento].ID_INC) < 0:
                continue
            
            if slot.tipoErogazione == TipoErogazione.Remoto:
                continue
            
            if slot.tipoLocale == TipoLocale.ACSLAB:
                self.listSlot_ACSLAB.append(slotId)
            
            
            elif slot.tipoLocale in [TipoLocale.Aula, TipoLocale.Aula_CA2, TipoLocale.Aula_CA, TipoLocale.Aula_TB, TipoLocale.Aula_WB]:
                capienzaAula:CapienzaLocale = getCapienzaFromNumStudenti(slot.numIscritti)
                
                if capienzaAula == CapienzaLocale.Media:
                    self.listSlot_AuleMedie.append(slotId)
                    self.listSlot_Aule_MedieMedioGrandi.append(slotId)
                    self.listSlot_Aule_PiccoleMedie.append(slotId)
                    self.listSlot_Aule_PiccoleMedieMedioGrandi.append(slotId)
                    self.listSlot_Aule_MedieMedioGrandiGrandi.append(slotId)           
                    if slot.preseElettriche == PreseElettriche.Si:
                        self.listSlot_PreseElettriche_AuleMedie.append(slotId)
                        self.listSlot_PreseElettriche_Aule_MedieMedioGrandi.append(slotId)
                        self.listSlot_PreseElettriche_Aule_PiccoleMedie.append(slotId)
                        self.listSlot_PreseElettriche_Aule_PiccoleMedieMedioGrandi.append(slotId)
                        self.listSlot_PreseElettriche_Aule_MedieMedioGrandiGrandi.append(slotId)                                        
                
                elif capienzaAula == CapienzaLocale.MedioGrande:
                    self.listSlot_AuleMedioGrandi.append(slotId)
                    self.listSlot_Aule_MedieMedioGrandi.append(slotId)
                    self.listSlot_Aule_MedioGrandiGrandi.append(slotId)
                    self.listSlot_Aule_PiccoleMedieMedioGrandi.append(slotId)  
                    self.listSlot_Aule_MedieMedioGrandiGrandi.append(slotId)   
                    if slot.preseElettriche == PreseElettriche.Si:
                        self.listSlot_PreseElettriche_AuleMedioGrandi.append(slotId)
                        self.listSlot_PreseElettriche_Aule_MedieMedioGrandi.append(slotId)
                        self.listSlot_PreseElettriche_Aule_MedioGrandiGrandi.append(slotId)
                        self.listSlot_PreseElettriche_Aule_PiccoleMedieMedioGrandi.append(slotId)  
                        self.listSlot_PreseElettriche_Aule_MedieMedioGrandiGrandi.append(slotId)                                                                                              
                
                elif capienzaAula == CapienzaLocale.Piccola:
                    self.listSlot_AulePiccole.append(slotId)
                    self.listSlot_Aule_PiccoleMedie.append(slotId)
                    self.listSlot_Aule_PiccoleMedieMedioGrandi.append(slotId)
                    if slot.preseElettriche == PreseElettriche.Si:
                        self.listSlot_PreseElettriche_AulePiccole.append(slotId)
                        self.listSlot_PreseElettriche_Aule_PiccoleMedie.append(slotId)
                        self.listSlot_PreseElettriche_Aule_PiccoleMedieMedioGrandi.append(slotId)                        
                
                elif capienzaAula == CapienzaLocale.Grande:
                    self.listSlot_AuleGrandi.append(slotId)
                    self.listSlot_Aule_MedioGrandiGrandi.append(slotId)
                    self.listSlot_Aule_MedieMedioGrandiGrandi.append(slotId)
                    if slot.preseElettriche == PreseElettriche.Si:
                        self.listSlot_PreseElettriche_AuleGrandi.append(slotId)
                        self.listSlot_PreseElettriche_Aule_MedioGrandiGrandi.append(slotId)
                        self.listSlot_PreseElettriche_Aule_MedieMedioGrandiGrandi.append(slotId)                                                                                 
                else:
                    self.log.error_log("LocaliHandler.createListsLocaliOfSlot(): capienzaAule sconosciuta")

            elif slot.tipoLocale == TipoLocale.AulaAttrezzata:
                self.log.error_log("LocaliHandler.createListsLocaliOfSlot(): TipoLocale deprecated")
            elif slot.tipoLocale == TipoLocale.AulaMultimediale:
                self.log.error_log("LocaliHandler.createListsLocaliOfSlot(): TipoLocale deprecated")
            elif slot.tipoLocale == TipoLocale.LABINF:
                self.listSlot_LABINF.append(slotId)
            elif slot.tipoLocale == TipoLocale.Laboratorio:
                self.listSlot_Laboratorio.append(slotId)
            elif slot.tipoLocale == TipoLocale.LADISPE:
                self.listSlot_LADISPE.append(slotId)
            elif slot.tipoLocale == TipoLocale.LAIB:
                self.listSlot_LAIB.append(slotId)
            elif slot.tipoLocale == TipoLocale.LED:
                self.listSlot_LED.append(slotId)
            elif slot.tipoLocale == TipoLocale.LED1:
                self.listSlot_LED1.append(slotId)
            elif slot.tipoLocale == TipoLocale.LED2:
                self.listSlot_LED2.append(slotId)     
            elif slot.tipoLocale == TipoLocale.Aula5T:
                self.listSlot_Aula5T.append(slotId) 
        
        if self.stats:
            self.printStatisticheCapienzaAule()
        
    def printStatisticheCapienzaAule(self):
        '''debug stats sulla quantità di Slot appartenenti ad un tipo di capienza particolare.\n
        NB: può capitare di avere soglie > 1, ciò significherebbe che dovrei allocare più Slot di quanti le Aule disponibili ne consentano;
        in realtà usando la features degli extension levels è possibile avere soglie > 1 ma al contempo essere di fronte ad una soluzione valida.\n
        Tuttavia da prove sperimentali si consiglia di non utilizzare mai dataset di partenza che portano ad avere soglie di occupazione
        anche solamente vicine a 1, perchè i tempi di esecuzione si allungherebbero enormemente.\n
        Si può apprezzare l'uso della features degli extension level andando ad osservare le stats di una soluzione, in cui può appunto capitare
        di trovarsi di fronte a soglie di occupazione > 1.'''
        self.log.info_log("")
        self.log.info_log("LocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza Piccola:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_AulePiccole)))          
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_AulePiccole)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Piccola)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))
        
        for capienza in [CapienzaLocale.Piccola, CapienzaLocale.Media, CapienzaLocale.MedioGrande, CapienzaLocale.Grande]:
            for day in range(self.AUX.get_NUM_DAY()):
                for hour in range(self.AUX.NUM_SLOT_PER_DAY):
                    slotRes:List[int] = list()
                    for slotId in range(len(self.AUX.pianoAllocazione)):
                        slot = self.AUX.pianoAllocazione[slotId]
                        if slot.tipoSlot == TipoSlot.SlotScelto and slot.daySlotAllocato == day and slot.hourSlotAllocato <= hour and slot.hourSlotAllocato + slot.numSlot > hour:
                            if slot.tipoLocale in [TipoLocale.Aula, TipoLocale.Aula_CA, TipoLocale.Aula_CA2, TipoLocale.Aula_TB, TipoLocale.Aula_WB]:
                                if getCapienzaFromNumStudenti(slot.numIscritti) == capienza:
                                    slotRes.append(slotId)
                    if len(slotRes) > self.getLimit(day,hour,TipoLocale.Aula,capienza)*1:
                        self.log.info_logUtils("\nSlot scelti capienza {}:".format(getStringFromCapienzaLocale(capienza)))                        
                        for slotId in slotRes:
                            res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                            res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                            self.log.info_logUtils(res)
                        self.log.info_logUtils("")
                        
        for day in range(self.AUX.get_NUM_DAY()):
            for hour in range(self.AUX.NUM_SLOT_PER_DAY):
                slotRes:List[int] = list()
                for slotId in range(len(self.AUX.pianoAllocazione)):
                    slot = self.AUX.pianoAllocazione[slotId]
                    if slot.tipoSlot == TipoSlot.SlotScelto and slot.daySlotAllocato == day and slot.hourSlotAllocato <= hour and slot.hourSlotAllocato + slot.numSlot > hour:
                        if slot.tipoLocale in [TipoLocale.Aula, TipoLocale.Aula_CA, TipoLocale.Aula_CA2, TipoLocale.Aula_TB, TipoLocale.Aula_WB]:
                                if getCapienzaFromNumStudenti(slot.numIscritti) in [CapienzaLocale.MedioGrande, CapienzaLocale.Grande]:
                                    slotRes.append(slotId)
                    if len(slotRes) > self.getLimit(day,hour,TipoLocale.Aula,capienza)*1:
                        self.log.info_logUtils("\nSlot scelti capienza MedioGrande e Grande:")                        
                        for slotId in slotRes:
                            res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                            res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                            self.log.info_logUtils(res)
                        self.log.info_logUtils("")
                    
        
        if sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Piccola, PreseElettriche.Si)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
            ) != 0:
            self.log.info_log("LocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza Piccola - Prese elettriche:")
            self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_PreseElettriche_AulePiccole)))          
            count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_PreseElettriche_AulePiccole)
            self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
            self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                            sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Piccola, PreseElettriche.Si)
                                for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                            )))        
            
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza Media:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_AuleMedie)))          
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_AuleMedie)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Media)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))
        
        if sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Media, PreseElettriche.Si)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
            ) != 0:
            self.log.info_log("LocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza Media - Prese elettriche:")
            self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_PreseElettriche_AuleMedie)))          
            count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_PreseElettriche_AuleMedie)
            self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
            self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                            sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Media, PreseElettriche.Si)
                                for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                            )))        
            
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza MedioGrande:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_AuleMedioGrandi)))          
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_AuleMedioGrandi)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.MedioGrande)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))
        
        if sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.MedioGrande, PreseElettriche.Si)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
            ) != 0:
            self.log.info_log("LocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza MedioGrande - Prese elettriche:")
            self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_PreseElettriche_AuleMedioGrandi)))          
            count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_PreseElettriche_AuleMedioGrandi)
            self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
            self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                            sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.MedioGrande, PreseElettriche.Si)
                                for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                            )))        
                              
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza Grande:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_AuleGrandi)))          
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_AuleGrandi)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Grande)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        ))) 
        
        if sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Grande, PreseElettriche.Si)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
            ) != 0:
            self.log.info_log("LocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza Grande - Prese elettriche:")
            self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_PreseElettriche_AuleGrandi)))          
            count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_PreseElettriche_AuleGrandi)
            self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
            self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                            sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Grande, PreseElettriche.Si)
                                for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                            )))                      
        
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche Aule Piccole + Aule Medie:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_Aule_PiccoleMedie)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_Aule_PiccoleMedie)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        nDisp:int = sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Media)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count / nDisp))    
        
        nDisp:int = sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Media,PreseElettriche.Si)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY))
        if nDisp != 0:
            self.log.info_log("LocaliHandler.createListsLocaliOfSlot(): statistiche Aule Piccole + Aule Medie - Prese elettriche:")
            self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_PreseElettriche_Aule_PiccoleMedie)))
            count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_PreseElettriche_Aule_PiccoleMedie)
            self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
            self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count / nDisp))              
        
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza MedioGrande + Grande:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_Aule_MedioGrandiGrandi)))          
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_Aule_MedioGrandiGrandi)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Grande)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))        
        
        if sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Grande,PreseElettriche.Si)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
            ) != 0:
            self.log.info_log("LocaliHandler.createListsLocaliOfSlot(): statistiche Aule di capienza MedioGrande + Grande - Prese elettriche:")
            self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_PreseElettriche_Aule_MedioGrandiGrandi)))          
            count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_PreseElettriche_Aule_MedioGrandiGrandi)
            self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
            self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                            sum(self.getLimit(day,hour,TipoLocale.Aula,CapienzaLocale.Grande,PreseElettriche.Si)
                                for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                            )))                
        
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche LABINF:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_LABINF)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_LABINF)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.LABINF,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))
        listRes:List[str] = list()
        for slotId in self.listSlot_LABINF:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti LABINF:")
            for res in listRes:
                self.log.info_log(res)
        
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche ACSLAB:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_ACSLAB)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_ACSLAB)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.ACSLAB,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))
        listRes:List[str] = list()
        for slotId in self.listSlot_ACSLAB:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti ACSLAB:")
            for res in listRes:
                self.log.info_log(res)        

        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche Aula5T:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_Aula5T)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_Aula5T)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.Aula5T,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        ))) 
        listRes:List[str] = list()
        for slotId in self.listSlot_Aula5T:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti Aula5T:")
            for res in listRes:
                self.log.info_log(res)           

        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche LADISPE:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_LADISPE)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_LADISPE)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.LADISPE,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))          
        listRes:List[str] = list()
        for slotId in self.listSlot_LADISPE:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti LADISPE:")
            for res in listRes:
                self.log.info_log(res)                   
        
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche LAIB:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_LAIB)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_LAIB)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.LAIB,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))          
        listRes:List[str] = list()
        for slotId in self.listSlot_LAIB:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti LAIB:")
            for res in listRes:
                self.log.info_log(res)            
        
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche LED:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_LED)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_LED)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.LED,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))      
        listRes:List[str] = list()
        for slotId in self.listSlot_LED:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti LED:")
            for res in listRes:
                self.log.info_log(res)               
       
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche LED1:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_LED1)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_LED1)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.LED1,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))   
        listRes:List[str] = list()
        for slotId in self.listSlot_LED1:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti LED1:")
            for res in listRes:
                self.log.info_log(res)        
        
        self.log.info_log("\nLocaliHandler.createListsLocaliOfSlot(): statistiche LED2:")
        self.log.info_log("Slot (lezioni) da assegnare: " + str(len(self.listSlot_LED2)))
        count:int = sum(self.AUX.pianoAllocazione[i].numSlot for i in self.listSlot_LED2)
        self.log.info_log("Slot (blocchi da 1.5h) da assegnare: " + str(count))
        self.log.info_log("Soglia di occupazione (media): {:.2f}".format(count /
                        sum(self.getLimit(day,hour,TipoLocale.LED2,CapienzaLocale.NonDisponibile)
                            for day in range(self.AUX.get_NUM_DAY()) for hour in range(self.AUX.NUM_SLOT_PER_DAY)
                        )))            
        listRes:List[str] = list()
        for slotId in self.listSlot_LED2:
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_giorno:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(self.AUX.pianoAllocazione[slotId].daySlotAllocato, -1, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)
            if self.AUX.pianoAllocazione[slotId].tipoSlot == TipoSlot.SlotScelto_fasciaOraria:
                res:str = self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].ID_INC + " - " + self.AUX.list_Insegnamenti[self.AUX.pianoAllocazione[slotId].idInsegnamento].titolo
                res += " " + getSlotFromDayHour(-1, self.AUX.pianoAllocazione[slotId].hourSlotAllocato, self.AUX.pianoAllocazione[slotId].numSlot)
                listRes.append(res)                                                
        if len(listRes) > 0:
            self.log.info_log("SlotScelti LED2:")
            for res in listRes:
                self.log.info_log(res)                                         
        
        
    def model_addRules(self, X_d, X_h, model:CpoModel):
        '''Crea tutti i constraint da aggiungere al modello e li ritorna (lista)'''
        eqs = list()
        
        for day in range(self.AUX.get_NUM_DAY()):
            for h in range(self.AUX.NUM_SLOT_PER_DAY):
                # (X_h[slotId] <= h) -> deve cominciare nello slot h o prima
                # ((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) -> deve finire nello slot attuale o dopo                
                
                eq1 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_ACSLAB) <= self.getLimit(day, h, TipoLocale.ACSLAB)
                if len(self.listSlot_ACSLAB) > 0:
                    eqs.append(eq1)
                
                eq2 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_LABINF) <= self.getLimit(day, h, TipoLocale.LABINF)
                if len(self.listSlot_LABINF) > 0:
                    eqs.append(eq2)
               
                # eq3 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                #           for slotId in self.listSlot_Laboratorio) <= self.getLimit(day, h, TipoLocale.Laboratorio, CapienzaLocale.NonDisponibile)
                
                eq4 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_LADISPE) <= self.getLimit(day, h, TipoLocale.LADISPE)
                if len(self.listSlot_LADISPE) > 0:
                    eqs.append(eq4)
                
                eq5 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_LAIB) <= self.getLimit(day, h, TipoLocale.LAIB)
                if len(self.listSlot_LAIB) > 0:
                    eqs.append(eq5)
                
                eq6 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_LED) <= self.getLimit(day, h, TipoLocale.LED)
                if len(self.listSlot_LED) > 0:
                    eqs.append(eq6)
                
                eq7 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_LED1) <= self.getLimit(day, h, TipoLocale.LED1) 
                if len(self.listSlot_LED1) > 0:
                    eqs.append(eq7)
                
                eq8 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_LED2) <= self.getLimit(day, h, TipoLocale.LED2)
                if len(self.listSlot_LED2) > 0:
                    eqs.append(eq8)
                
                # Aule Piccole
                listEq = list()
                eq11_a = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_AulePiccole) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.Piccola)
                listEq.append(eq11_a)
                if self.PARAM.capienzaAuleExtended >= 1:
                    eq11_b = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_Aule_PiccoleMedie) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.Media)
                    listEq.append(eq11_b)
                if self.PARAM.capienzaAuleExtended >= 2:
                    eq11_c = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_Aule_PiccoleMedieMedioGrandi) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.MedioGrande)                    
                    listEq.append(eq11_c)
                if self.PARAM.capienzaAuleExtended >= 3:
                    self.log.warning_log("LocaliHandler.model_addRules(): Parameters.capienzaAuleExtended >= 3 not yet implemented")
                if len(listEq) > 1:
                    eq11 = rOr(listEq)
                else:
                    eq11 = eq11_a
                if len(self.listSlot_AulePiccole) > 0:
                    eqs.append(eq11)
                    
                # Aule Piccole con prese elettriche
                listEq = list()
                eq11_a_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_AulePiccole) <= self.getLimit(day, h, TipoLocale.Aula,
                                                                                CapienzaLocale.Piccola, PreseElettriche.Si)
                listEq.append(eq11_a_prese)
                if self.PARAM.capienzaAuleExtended >= 1:
                    eq11_b_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_Aule_PiccoleMedie) <= self.getLimit(day, h, TipoLocale.Aula,
                                                                                CapienzaLocale.Media, PreseElettriche.Si)
                    listEq.append(eq11_b_prese)
                if self.PARAM.capienzaAuleExtended >= 2:
                    eq11_c_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_Aule_PiccoleMedieMedioGrandi) <= self.getLimit(day, h, TipoLocale.Aula,
                                                                                CapienzaLocale.MedioGrande, PreseElettriche.Si)                    
                    listEq.append(eq11_c_prese)
                if self.PARAM.capienzaAuleExtended >= 3:
                    self.log.warning_log("LocaliHandler.model_addRules(): Parameters.capienzaAuleExtended >= 3 not yet implemented")
                if len(listEq) > 1:
                    eq11_prese = rOr(listEq)
                else:
                    eq11_prese = eq11_a_prese    
                if len(self.listSlot_PreseElettriche_AulePiccole) > 0:
                    eqs.append(eq11_prese)                
            
                # Aule MedioPiccole
                # listEq = list()
                # eq12_a = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                #           for slotId in self.listSlot_AuleMedioPiccole) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.MedioPiccola)
                # if self.PARAM.capienzaAuleExtended >= 1:
                #     eq12_b = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                #           for slotId in self.listSlot_Aule_MedioPiccoleMedie) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.Media)
                #     listEq.append(eq12_b)
                # if self.PARAM.capienzaAuleExtended >= 2:
                #     eq12_c = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                #           for slotId in self.listSlot_Aule_MedioPiccoleMedieMedioGrandi) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.MedioGrande)
                #     listEq.append(eq12_c)
                # if self.PARAM.capienzaAuleExtended >= 3:
                #     self.log.warning_log("LocaliHandler.model_addRules(): Parameters.capienzaAuleExtended >= 3 not yet implemented")                    
                # if len(listEq) > 1:
                #     eq12 = rOr(listEq)
                # else:
                #     eq12 = eq12_a                      
                         
                # Aule Medie
                listEq = list()
                eq9_a = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_AuleMedie) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.Media)
                if self.PARAM.capienzaAuleExtended >= 1:
                    eq9_b = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_Aule_MedieMedioGrandi) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.MedioGrande)
                    listEq.append(eq9_b)
                if self.PARAM.capienzaAuleExtended >= 2:
                    eq9_c = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_Aule_MedieMedioGrandiGrandi) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.Grande)
                    listEq.append(eq9_c)
                if self.PARAM.capienzaAuleExtended >= 3 and self.debug:
                    self.log.info_log("LocaliHandler.model_addRules(): Parameters.capienzaAuleExtended >= 3 for Capienza.Media")                    
                if len(listEq) > 1:
                    eq9 = rOr(listEq)
                else:
                    eq9 = eq9_a   
                if len(self.listSlot_AuleMedie) > 0:
                    eqs.append(eq9)
                    
                # Aule Medie con prese elettriche
                listEq = list()
                eq9_a_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_AuleMedie) <= self.getLimit(day, h, TipoLocale.Aula, 
                                                                                            CapienzaLocale.Media, PreseElettriche.Si)
                if self.PARAM.capienzaAuleExtended >= 1:
                    eq9_b_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_Aule_MedieMedioGrandi) <= self.getLimit(day, h, TipoLocale.Aula, 
                                                                                            CapienzaLocale.MedioGrande, PreseElettriche.Si)
                    listEq.append(eq9_b_prese)
                if self.PARAM.capienzaAuleExtended >= 2:
                    eq9_c_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_Aule_MedieMedioGrandiGrandi) <= self.getLimit(day, h, TipoLocale.Aula,
                                                                                            CapienzaLocale.Grande, PreseElettriche.Si)
                    listEq.append(eq9_c_prese)
                if self.PARAM.capienzaAuleExtended >= 3 and self.debug:
                    self.log.info_log("LocaliHandler.model_addRules(): Parameters.capienzaAuleExtended >= 3 for Capienza.Media")                    
                if len(listEq) > 1:
                    eq9_prese = rOr(listEq)
                else:
                    eq9_prese = eq9_a_prese                                   
                if len(self.listSlot_PreseElettriche_AuleMedie) > 0:
                    eqs.append(eq9_prese)
                
                # Aule MedioGrandi
                listEq = list()
                eq10_a = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_AuleMedioGrandi) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.MedioGrande)
                if self.PARAM.capienzaAuleExtended >= 1:
                    eq10_b = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_Aule_MedioGrandiGrandi) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.Grande)
                    listEq.append(eq10_b)
                if self.PARAM.capienzaAuleExtended >= 2 and self.debug:
                    self.log.info_log("LocaliHandler.model_addRules(): Parameters.capienzaAuleExtended >= 2 for Capienza.MedioGrande")                    
                if len(listEq) > 1:
                    eq10 = rOr(listEq)
                else:
                    eq10 = eq10_a 
                if len(self.listSlot_AuleMedioGrandi) > 0:
                    eqs.append(eq10)
                    
                # Aule MedioGrandi con prese elettriche
                listEq = list()
                eq10_a_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_AuleMedioGrandi) <= self.getLimit(day, h, TipoLocale.Aula, 
                                                                                            CapienzaLocale.MedioGrande, PreseElettriche.Si)
                if self.PARAM.capienzaAuleExtended >= 1:
                    eq10_b_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_Aule_MedioGrandiGrandi) <= self.getLimit(day, h, TipoLocale.Aula, 
                                                                                            CapienzaLocale.Grande, PreseElettriche.Si)
                    listEq.append(eq10_b_prese)
                if self.PARAM.capienzaAuleExtended >= 2 and self.debug:
                    self.log.info_log("LocaliHandler.model_addRules(): Parameters.capienzaAuleExtended >= 2 for Capienza.MedioGrande")                    
                if len(listEq) > 1:
                    eq10_prese = rOr(listEq)
                else:
                    eq10_prese = eq10_a_prese       
                if len(self.listSlot_PreseElettriche_AuleMedioGrandi) > 0:
                    eqs.append(eq10_prese)

                # Aule Grandi    
                eq13 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_AuleGrandi) <= self.getLimit(day, h, TipoLocale.Aula, CapienzaLocale.Grande)
                if len(self.listSlot_AuleGrandi) > 0:
                    eqs.append(eq13)
                
                # Aule Grandi con prese elettriche  
                eq13_prese = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_PreseElettriche_AuleGrandi) <= self.getLimit(day, h, TipoLocale.Aula, 
                                                                                            CapienzaLocale.Grande, PreseElettriche.Si)                
                if len(self.listSlot_PreseElettriche_AuleGrandi) > 0:
                    eqs.append(eq13_prese)
                
                # Aula 5T
                eq14 = model.sum((X_d[slotId] == day)*(X_h[slotId] <= h)*((X_h[slotId]+self.AUX.pianoAllocazione[slotId].numSlot > h)) 
                          for slotId in self.listSlot_Aula5T) <= self.getLimit(day, h, TipoLocale.Aula5T)
                if len(self.listSlot_Aula5T) > 0:
                    eqs.append(eq14)
                                    
        if self.debug:
            self.log.info_log(eq1)     
            self.log.info_log(eq11)
            self.log.info_log(eq11_prese)
        
        return eqs


